<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\User\Block;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\User\UserBlockTable;
use Erlage\Photogram\Data\Models\User\Block\UserBlockModel;

final class UserBlockContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $offset = self::$request -> findKeyOffset(UserBlockTable::ID, UserBlockTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $userBlockTableQuery = (new Query()) -> from(UserBlockTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select users blocked by current user
            |--------------------------------------------------------------------------
            */

            $userBlockTableQuery -> where(UserBlockTable::BLOCKED_BY_USER_ID, self::$authedUserModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $userBlockTableQuery -> greaterThan(UserBlockTable::ID, $offset);
            }
            else
            {
                $userBlockTableQuery -> lessThan(UserBlockTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by & limit
            |--------------------------------------------------------------------------
            */

            $userBlockTableQuery
                -> orderByDesc(UserBlockTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_USER_BLOCK));

            /*
            |--------------------------------------------------------------------------
            | get beans
            |--------------------------------------------------------------------------
            */

            $userBlockBeans = $userBlockTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($userBlockBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | else add to response
            |--------------------------------------------------------------------------
            */

            self::processBeans(UserBlockTable::getTableName(), $userBlockBeans, function (UserBlockModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's models are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getBlockedUserId());

                /*
                |--------------------------------------------------------------------------
                | add block model to additional container as well
                |--------------------------------------------------------------------------
                */

                self::addToAdditionalResponse(UserBlockTable::getTableName(), $model -> getDataMap());
            });
        });
    }
}
